<?php

namespace Modules\Album\app\Http\Controllers\Dashboard\Frest;


use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Barryvdh\DomPDF\Facade\Pdf;
use Modules\Album\app\Models\Album;
use Modules\Album\app\Http\Requests\Dashboard\Frest\Album\StoreAlbumRequest;
use Modules\Album\app\Http\Requests\Dashboard\Frest\Album\UpdateAlbumRequest;
use Modules\Category\app\Models\Category;
use Modules\Core\app\Http\Controllers\Helpers\File;
use Modules\Core\app\Http\Controllers\Helpers\Str;
use Modules\Field\app\Traits\FieldCustom;
use Modules\Tag\app\Models\Tag;

class AlbumController extends Controller
{
    use FieldCustom;
    public function index()
    {

        $this->authorize('album.access');

        $user = Auth::guard('admin')->user();
        $categories = Category::whereState(1)->whereType('albums')->whereHas('roles', function ($query)  use ($user) {
             $query->whereIn('id',$user->roles->pluck('role_id'));
        })->get();


        $query = Album::
        when(!is_null(Request()->q),function ($query){
            return $query->where('title','LIKE','%'.Request()->q."%")
                ->orWhere('id','LIKE','%'.Request()->q."%");
        })
            ->when(!is_null(Request()->cat_id),function ($query){
                return $query->whereIn('cat_id',Request()->cat_id);
            })
            ->when(!is_null(Request()->state),function ($query){
                return $query->whereIn('state',Request()->state);
            })
            ->when(!is_null(Request()->categories),function ($query){
                return $query->whereHas('categories',function ($query){
                    $query->whereIn('id',Request()->categories);
                });
            })
            ->when(!is_null(Request()->start_date), function ($query) {
                $start_date = Str::convertToGregorian(Request()->start_date . ':00');
                return $query->whereDate('created_at','>=',$start_date);
            })
            ->when(!is_null(Request()->end_date), function ($query)  {
                $end_date = Str::convertToGregorian(Request()->end_date . ':00');
                return $query->whereDate('created_at','<=',$end_date);
            })
            ->with('categories','category')
            ->latest('id');




        if(isset($_GET['print'])){
            $album = $query->limit(200)->get();
            return view('template.print.album' , compact('album'));
        }

        if(isset($_GET['pdf'])){
            $album = $query->limit(200)->get();
            Pdf::setOption(['dpi' => 150, 'defaultFont' => 'sans-serif']);
            $pdf = Pdf::loadView('template.print.album', ['album' => $album]);
            return $pdf->download('album.pdf');
        }

        if (isset($_GET['excel'])) {
            $items = $query->latest()->get();
            return Excel::download(new NewsExport($items),'album.xlsx');
        }

        $albums = $query->paginate(15);



        return view('album::Dashboard.Frest.album.index' , compact('albums','categories'));


    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->authorize('album.create');

        $user = Auth::guard('admin')->user();
        $categories = Category::whereState(1)->whereType('albums')->whereState(1)->whereHas('roles', function ($query)  use ($user) {
             $query->whereIn('id',$user->roles->pluck('role_id'));
        })->whereNull('parent_id')->get();

        $tags = Tag::all();
        $album = [];

        return view('album::Dashboard.Frest.album.create',compact('tags','album','categories'));

    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreAlbumRequest $request)
    {
        $this->authorize('album.create');

        $image = File::uploadByFileManager($request->image);


        $publish_up = null; $publish_down=null;
        if($request->publish_up != null)
        {
            $publish_up = Str::convertToGregorian($request->publish_up . ':00');
        }

        if($request->publish_down != null)
        {
            $publish_down = Str::convertToGregorian($request->publish_down . ':00');
        }

        $album = Album::create([
            'title' => $request->title,
            'body' => $request->body,
            'image' => $image,
            'link' => $request->link,
            'state' => $request->state,
            'publish_up' => $publish_up,
            'publish_down' => $publish_down,
            'main_cat' => $request->main_cat,
        ]);



        $tags = json_decode($request->tag);

        $tags_id = [];
        if($tags != null){
            foreach($tags as $tag){
                $tag = Tag::firstOrCreate(['name' => $tag->value]);
                array_push($tags_id,$tag->id);
            }
            $album->tags()->attach($tags_id);
        }

        $album->categories()->attach($request->categories);




        $this->storeCustomField($album,$request->fields);


        if(isset($_POST['save']))
        {
            Session::flash('success','با موفقیت انجام شد');
            return redirect()->route('admin.album.index');
        }

        return redirect()->route('admin.album.edit',$album->id);





    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $this->authorize('album.edit');

        $album = Album::whereId($id)->with('categories')->first();

        $user = Auth::guard('admin')->user();

        $categories = Category::whereState(1)->whereType('albums')->whereState(1)->whereHas('roles', function ($query)  use ($user) {
             $query->whereIn('id',$user->roles->pluck('role_id'));
        })->whereNull('parent_id')->get();

        $tags = Tag::all();


        return view('album::Dashboard.Frest.album.edit',compact('album','categories','tags'));




    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateAlbumRequest $request, $id)
    {
        $this->authorize('album.edit');


        $album = Album::whereId($id)->first();


        $image = $album->image;
        if($request->image != null) {
            $image = File::uploadByFileManager($request->image);
        }


        $publish_up = $album->publish_up; $publish_down = $album->publish_down;
        if($request->publish_up != null)
        {
            $publish_up = Str::convertToGregorian($request->publish_up . ':00');
        }

        if($request->publish_down != null)
        {
            $publish_down = Str::convertToGregorian($request->publish_down . ':00');
        }

        $album->title = $request->title;
        $album->body = $request->body;
        $album->image = $image;
        $album->link = $request->link;
        $album->state = $request->state;
        $album->publish_up = $publish_up;
        $album->publish_down = $publish_down;
        $album->main_cat = $request->main_cat;
        $album->update();



        $tags = json_decode($request->tag);
        $tags_id = [];
        if($tags != null){
            foreach($tags as $tag){
                $tag = Tag::firstOrCreate(['name' => $tag->value]);
                array_push($tags_id,$tag->id);
            }
            $album->tags()->sync($tags_id);
        }

        $album->categories()->sync($request->categories);

        $this->storeCustomField($album,$request->fields);

        if(isset($_POST['save']))
        {
            Session::flash('success','با موفقیت انجام شد');
            return redirect()->route('admin.album.index');
        }

        return redirect()->route('admin.album.edit',$album->id);






    }




    public function destroy($id)
    {
        $this->authorize('album.delete');


        $album = Album::find($id);
        $album->delete();


        session::flash('success','با موفقیت انجام شد');
        return redirect()->back();

    }


    public function bulkAction()
    {

        $items = [];
        $items = explode(',',Request()->items);



        if(isset($_GET['delete']))
        {
            $this->authorize('album.delete');
            Album::whereIn('id',$items)->delete();
        }
        if (isset($_GET['unPublish']))
        {
            $this->authorize('album.edit');
            Album::whereIn('id',$items)->update([
                'state' => 0
            ]);
        }
        if(isset($_GET['publish']))
        {
            $this->authorize('album.edit');
            Album::whereIn('id',$items)->update([
                'state' => 1
            ]);
        }



        session::flash('success','با موفقیت انجام شد');
        return redirect()->back();

    }
}
