<?php

namespace Modules\Core\app\Http\Controllers\Helpers;

use Exception;
use Illuminate\Http\UploadedFile;
use Intervention\Image\Facades\Image;
use Log;
use Storage;

class File
{
    /**
     * Upload an image with specific width and height.
     *
     * @param UploadedFile $image
     * @param string $dir
     * @param int $width
     * @param int $height
     * @param string $disk
     * @return string|boolean
     */
    public static function imageUpload($image, $dir, $width = null, $height = null, $disk = 'public')
    {

        if(is_file($image))
        {
            return File::uploadImage($image, $dir, $width, $height, $disk);
        }
        else
        {
            return File::base64Upload($image, $dir, $disk);
        }
    }

    public static function uploadByFileManager($path = null,$default = null){



        if($path != null) {
            if(config('file-manager.diskList')[0] == 'public') {
                return $path;
            } else {
                return $path;
            }
        } else {
            return $default;
        }


    }


    /**
     * Upload an image with specific width and height.
     *
     * @param UploadedFile $image
     * @param string $dir
     * @param int $width
     * @param int $height
     * @param string $disk
     * @return string|boolean
     */
    public static function uploadImage($image, $dir, $width = null, $height = null, $disk = 'public')
    {
        try {
            $path = $image->hashName($dir);
            $img = Image::make($image);
            if ($width && $height) {
                $img->fit($width, $height, function ($constraint) {
                    $constraint->aspectRatio();
                    //$constraint->upsize();
                });
            }
            Storage::disk($disk)->put($path, (string) $img->encode());
            return $path;

        } catch (Exception $e) {
            Log::error('Failed image upload: ' . $e->getMessage());
        }
        return false;
    }

    /**
     * Upload a file.
     *
     * @param UploadedFile $file
     * @param string $dir
     * @param null $disk
     * @return bool
     */
    public static function fileUpload($file, $dir, $disk = null)
    {
        try {
            if ($disk) {
                $path = $file->store($dir, $disk);
            } else {
                $path = $file->store($dir);
            }
            return $path;
        } catch (Exception $e) {
            Log::error('File upload error: ' . $e->getMessage());
        }
        return false;
    }

    /**
     * Delete permanently a file from disk.
     *
     * @param string $path
     * @param string $disk
     * @return array
     */
    public static function delete($path, $disk = 'public')
    {
        $output = [
            'success' => false,
            'message' => 'File not exists'
        ];

        if (Storage::disk($disk)->exists($path)) {
            try {

                Storage::disk($disk)->delete($path);

                $output = [
                    'success' => true
                ];

            } catch (Exception $e) {

                $output = [
                    'success' => false,
                    'message' => $e->getMessage()
                ];
            }
        }

        return $output;
    }

    /**
     * Insert an image as image watermark.
     *
     * @param string $image
     * @param string $watermark
     * @param string $position
     */
    public static function insertWatermark($image, $watermark, $position = 'bottom-right')
    {
        $img = Image::make($image);
        $width = $img->width();
        $height = $img->height();
        $watermarkWidth = ($width / 100) * 20;
        $watermarkHeight = ($height / 100) * 20;
        $watermarkImg = Image::make($watermark);
        $resizedWatermark = $watermarkImg->resize($watermarkWidth, $watermarkHeight);
        $img->insert($resizedWatermark, $position, 10, 10);
        $img->save($image);
    }

    /**
     * Base64 file upload.
     *
     * @param string $base64Image
     * @param string $dir
     * @param string $disk
     * @return string
     */
    public static function base64Upload($base64Image, $dir, $disk = 'public')
    {
        @list($type, $file_data) = explode(';', $base64Image);
        @list(, $file_data) = explode(',', $file_data);
        $extension = explode('/', $type)[1];
        $imageName = time() . '.' . $extension;
        $path = "$dir/{$imageName}";
        Storage::disk($disk)->put($path, base64_decode($file_data));

        return $path;
    }


    public static function getSrc($file){
        if($file == null){
            return config('app.APP_URL').'/no-image.png';
        }

        if(config('file-manager.diskList')[0] == 'public') {
            return config('app.APP_URL').'/storage/'.$file;
        }

        if(config('file-manager.diskList')[0] == 'ftp') {
            return env('FTP_URL').'/'.$file;
        }
    }
}
