<?php

namespace Modules\Video\app\Http\Controllers\Dashboard\Frest;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;

use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Category\app\Models\Category;
use Modules\Core\app\Http\Controllers\Helpers\File;
use Modules\Core\app\Http\Controllers\Helpers\Str;
use Modules\Field\app\Traits\FieldCustom;
use Modules\Tag\app\Models\Tag;
use Modules\Video\app\Models\Video;
use Modules\Video\app\Http\Requests\Dashboard\Frest\Video\StoreVideoRequest;
use Modules\Video\app\Http\Requests\Dashboard\Frest\Video\UpdateVideoRequest;

class VideoController extends Controller
{
    use FieldCustom;
    public function index()
    {

        $this->authorize('video.access');

        $user = Auth::guard('admin')->user();
        $categories = Category::whereState(1)->whereType('videos')->whereHas('roles', function ($query)  use ($user) {
             $query->whereIn('id',$user->roles->pluck('role_id'));
        })->get();


        $query = Video::
        when(!is_null(Request()->q),function ($query){
            return $query->where('title','LIKE','%'.Request()->q."%")
                ->orWhere('id','LIKE','%'.Request()->q."%");
        })
            ->when(!is_null(Request()->cat_id),function ($query){
                return $query->whereIn('cat_id',Request()->cat_id);
            })
            ->when(!is_null(Request()->state),function ($query){
                return $query->whereIn('state',Request()->state);
            })
            ->when(!is_null(Request()->categories),function ($query){
                return $query->whereHas('categories',function ($query){
                    $query->whereIn('id',Request()->categories);
                });
            })

            ->when(!is_null(Request()->start_date), function ($query) {
                $start_date = Str::convertToGregorian(Request()->start_date . ':00');
                return $query->whereDate('created_at','>=',$start_date);
            })
            ->when(!is_null(Request()->end_date), function ($query)  {
                $end_date = Str::convertToGregorian(Request()->end_date . ':00');
                return $query->whereDate('created_at','<=',$end_date);
            })
            ->with('categories','category')
            ->latest('id');




        if(isset($_GET['print'])){
            $video = $query->limit(200)->get();
            return view('template.print.video' , compact('video'));
        }

        if(isset($_GET['pdf'])){
            $video = $query->limit(200)->get();
            Pdf::setOption(['dpi' => 150, 'defaultFont' => 'sans-serif']);
            $pdf = Pdf::loadView('template.print.video', ['video' => $video]);
            return $pdf->download('video.pdf');
        }

        if (isset($_GET['excel'])) {
            $items = $query->latest()->get();
            return Excel::download(new NewsExport($items),'video.xlsx');
        }

        $videos = $query->paginate(15);

        return view('video::Dashboard.Frest.video.index' , compact('videos','categories'));


    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->authorize('video.create');

        $user = Auth::guard('admin')->user();
        $categories = Category::whereState(1)->whereType('videos')->whereState(1)->whereHas('roles', function ($query)  use ($user) {
             $query->whereIn('id',$user->roles->pluck('role_id'));
        })->whereNull('parent_id')->get();

        $tags = Tag::all();
        $video = [];

        return view('video::Dashboard.Frest.video.create',compact('tags','video','categories'));

    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreVideoRequest $request)
    {
        $this->authorize('video.create');

        $cover = File::uploadByFileManager($request->cover);
        $file = File::uploadByFileManager($request->file);

        $publish_up = null; $publish_down=null;
        if($request->publish_up != null)
        {
            $publish_up = Str::convertToGregorian($request->publish_up . ':00');
        }

        if($request->publish_down != null)
        {
            $publish_down = Str::convertToGregorian($request->publish_down . ':00');
        }

        $video = Video::create([
            'title' => $request->title,
            'body' => $request->body,
            'link' => $request->link,
            'cover' => $cover,
            'file' => $file,
            'state' => $request->state,
            'publish_up' => $publish_up,
            'publish_down' => $publish_down,
            'main_cat' => $request->main_cat,

        ]);



        $tags = json_decode($request->tag);

        $tags_id = [];
        if($tags != null){
            foreach($tags as $tag){
                $tag = Tag::firstOrCreate(['name' => $tag->value]);
                array_push($tags_id,$tag->id);
            }
            $video->tags()->attach($tags_id);
        }

        $video->categories()->attach($request->categories);



        $this->storeCustomField($video,$request->fields);



        if(isset($_POST['save']))
        {
            Session::flash('success','با موفقیت انجام شد');
            return redirect()->route('admin.video.index');
        }

        return redirect()->route('admin.video.edit',$video->id);





    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $this->authorize('video.edit');

        $video = Video::whereId($id)->with('categories')->first();

        $user = Auth::guard('admin')->user();

        $categories = Category::whereState(1)->whereType('videos')->whereState(1)->whereHas('roles', function ($query)  use ($user) {
             $query->whereIn('id',$user->roles->pluck('role_id'));
        })->whereNull('parent_id')->get();

        $tags = Tag::all();


        return view('video::Dashboard.Frest.video.edit',compact('video','categories','tags'));




    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateVideoRequest $request, $id)
    {
        $this->authorize('video.edit');


        $video = Video::whereId($id)->first();


        $cover = File::uploadByFileManager($request->cover);
        $file = File::uploadByFileManager($request->file);

        $publish_up = $video->publish_up; $publish_down = $video->publish_down;
        if($request->publish_up != null)
        {
            $publish_up = Str::convertToGregorian($request->publish_up . ':00');
        }

        if($request->publish_down != null)
        {
            $publish_down = Str::convertToGregorian($request->publish_down . ':00');
        }

       $video->title = $request->title;
       $video->body = $request->body;
       $video->cover = $cover == null ? $video->cover : $cover ;
       $video->file = $file == null ? $video->file : $file;
       $video->state = $request->state;
       $video->publish_up = $publish_up;
       $video->publish_down = $publish_down;
       $video->main_cat = $request->main_cat;

       $video->update();


        $this->storeCustomField($video,$request->fields);


        $tags = json_decode($request->tag);
        $tags_id = [];
        if($tags != null){
            foreach($tags as $tag){
                $tag = Tag::firstOrCreate(['name' => $tag->value]);
                array_push($tags_id,$tag->id);
            }
            $video->tags()->sync($tags_id);
        }

        $video->categories()->sync($request->categories);



        if(isset($_POST['save']))
        {
            Session::flash('success','با موفقیت انجام شد');
            return redirect()->route('admin.video.index');
        }

        return redirect()->route('admin.video.edit',$video->id);






    }




    public function destroy($id)
    {
        $this->authorize('video.delete');


        $video = Video::find($id);
        $video->delete();


        session::flash('success','با موفقیت انجام شد');
        return redirect()->back();

    }


    public function bulkAction()
    {

        $items = [];
        $items = explode(',',Request()->items);



        if(isset($_GET['delete']))
        {
            $this->authorize('video.delete');
            Video::whereIn('id',$items)->delete();
        }
        if (isset($_GET['unPublish']))
        {
            $this->authorize('video.edit');
            Video::whereIn('id',$items)->update([
                'state' => 0
            ]);
        }
        if(isset($_GET['publish']))
        {
            $this->authorize('video.edit');
            Video::whereIn('id',$items)->update([
                'state' => 1
            ]);
        }



        session::flash('success','با موفقیت انجام شد');
        return redirect()->back();

    }
}
